! $Id$
!
! Earth System Modeling Framework
! Copyright (c) 2002-2025, University Corporation for Atmospheric Research,
! Massachusetts Institute of Technology, Geophysical Fluid Dynamics
! Laboratory, University of Michigan, National Centers for Environmental
! Prediction, Los Alamos National Laboratory, Argonne National Laboratory,
! NASA Goddard Space Flight Center.
! Licensed under the University of Illinois-NCSA License.
!
!==============================================================================
#define ESMF_FILENAME "ESMF_FieldGet.F90"
!==============================================================================
!
! ESMF FieldGet module
module ESMF_FieldGetMod
!
!==============================================================================
!
! This file contains the FieldGet() methods.
!
!------------------------------------------------------------------------------
! INCLUDES
! < ignore blank lines below. they are created by the files which
! define various macros. >
#include "ESMF.h"


!------------------------------------------------------------------------------
! !USES:
  use ESMF_UtilTypesMod
  use ESMF_BaseMod
  use ESMF_VMMod
  use ESMF_LogErrMod
  use ESMF_ArraySpecMod
  use ESMF_LocalArrayMod
  use ESMF_ArrayMod
  use ESMF_ArrayGetMod
  use ESMF_DistGridMod
  use ESMF_GridMod
  use ESMF_XGridMod
  use ESMF_MeshMod
  use ESMF_LocStreamMod
  use ESMF_GeomMod
  use ESMF_StaggerLocMod
  use ESMF_InitMacrosMod

  use ESMF_FieldMod

  implicit none

!------------------------------------------------------------------------------
! !PRIVATE TYPES:
  private

!------------------------------------------------------------------------------
! !PUBLIC MEMBER FUNCTIONS:

  public ESMF_FieldGet
  public ESMF_FieldGetBounds
  public ESMF_FieldGetTimestamp

!------------------------------------------------------------------------------
! The following line turns the CVS identifier string into a printable variable.
  character(*), parameter, private :: version = &
    '$Id$'

!==============================================================================
!
! INTERFACE BLOCKS
!
!==============================================================================


!------------------------------------------------------------------------------
!BOPI
! !IROUTINE: ESMF_FieldGet -- Overloaded FieldGet interface

! !INTERFACE:
  interface ESMF_FieldGet

! !PRIVATE MEMBER FUNCTIONS:
!
    module procedure ESMF_FieldGetDefault
    !------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
#ifndef ESMF_NO_INTEGER_1_BYTE 
 module procedure ESMF_FieldGetDataPtr1DI1 
 module procedure ESMF_FieldGetDataPtr2DI1 
 module procedure ESMF_FieldGetDataPtr3DI1 
 module procedure ESMF_FieldGetDataPtr4DI1 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldGetDataPtr5DI1 
 module procedure ESMF_FieldGetDataPtr6DI1 
 module procedure ESMF_FieldGetDataPtr7DI1 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
 module procedure ESMF_FieldGetDataPtr1DI2 
 module procedure ESMF_FieldGetDataPtr2DI2 
 module procedure ESMF_FieldGetDataPtr3DI2 
 module procedure ESMF_FieldGetDataPtr4DI2 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldGetDataPtr5DI2 
 module procedure ESMF_FieldGetDataPtr6DI2 
 module procedure ESMF_FieldGetDataPtr7DI2 
#endif 
#endif 
 module procedure ESMF_FieldGetDataPtr1DI4 
 module procedure ESMF_FieldGetDataPtr1DI8 
 module procedure ESMF_FieldGetDataPtr1DR4 
 module procedure ESMF_FieldGetDataPtr1DR8 
 module procedure ESMF_FieldGetDataPtr2DI4 
 module procedure ESMF_FieldGetDataPtr2DI8 
 module procedure ESMF_FieldGetDataPtr2DR4 
 module procedure ESMF_FieldGetDataPtr2DR8 
 module procedure ESMF_FieldGetDataPtr3DI4 
 module procedure ESMF_FieldGetDataPtr3DI8 
 module procedure ESMF_FieldGetDataPtr3DR4 
 module procedure ESMF_FieldGetDataPtr3DR8 
 module procedure ESMF_FieldGetDataPtr4DI4 
 module procedure ESMF_FieldGetDataPtr4DI8 
 module procedure ESMF_FieldGetDataPtr4DR4 
 module procedure ESMF_FieldGetDataPtr4DR8 
#ifndef ESMF_NO_GREATER_THAN_4D 
 module procedure ESMF_FieldGetDataPtr5DI4 
 module procedure ESMF_FieldGetDataPtr5DI8 
 module procedure ESMF_FieldGetDataPtr5DR4 
 module procedure ESMF_FieldGetDataPtr5DR8 
 module procedure ESMF_FieldGetDataPtr6DI4 
 module procedure ESMF_FieldGetDataPtr6DI8 
 module procedure ESMF_FieldGetDataPtr6DR4 
 module procedure ESMF_FieldGetDataPtr6DR8 
 module procedure ESMF_FieldGetDataPtr7DI4 
 module procedure ESMF_FieldGetDataPtr7DI8 
 module procedure ESMF_FieldGetDataPtr7DR4 
 module procedure ESMF_FieldGetDataPtr7DR8 
#endif 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 


! !DESCRIPTION:
! This interface provides a single entry point for the various
! types of {\tt ESMF\_FieldGet} subroutines.
  end interface
!EOPI
!------------------------------------------------------------------------------


!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

contains

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!


!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldGetDefault"
!BOP
! !IROUTINE: ESMF_FieldGet - Get object-wide Field information

! !INTERFACE:
  ! Private name; call using ESMF_FieldGet()
  subroutine ESMF_FieldGetDefault(field, keywordEnforcer, arrayspec, status, &
    geomtype, geom, grid, mesh, locstream, xgrid, array, localarrayList, &
    typekind, rank, dimCount, geomDimCount, ungriddedDimCount, &
    replicatedDimCount, staggerloc, meshloc, xgridside, gridindex, &
    gridToFieldMap, ungriddedLBound, ungriddedUBound, &
    totalLWidth, totalUWidth, localDeCount, ssiLocalDeCount, &
    localDeToDeMap, minIndex, maxIndex, elementCount, &
    localMinIndex, localMaxIndex, localElementCount, isESMFAllocated, &
    name, vm, rc)
!
! !ARGUMENTS:
    type(ESMF_Field), intent(in) :: field
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    type(ESMF_ArraySpec), intent(out), optional :: arrayspec
    type(ESMF_FieldStatus_Flag),intent(out), optional :: status
    type(ESMF_GeomType_Flag), intent(out), optional :: geomtype
    type(ESMF_Geom), intent(out), optional :: geom
    type(ESMF_Grid), intent(out), optional :: grid
    type(ESMF_Mesh), intent(out), optional :: mesh
    type(ESMF_LocStream), intent(out), optional :: locstream
    type(ESMF_XGrid), intent(out), optional :: xgrid
    type(ESMF_Array), intent(out), optional :: array
    type(ESMF_LocalArray), target, intent(out), optional :: localarrayList(:)
    type(ESMF_TypeKind_Flag), intent(out), optional :: typekind
    integer, intent(out), optional :: rank
    integer, intent(out), optional :: dimCount
    integer, intent(out), optional :: geomDimCount
    integer, intent(out), optional :: ungriddedDimCount
    integer, intent(out), optional :: replicatedDimCount
    type(ESMF_StaggerLoc), intent(out), optional :: staggerloc
    type(ESMF_MeshLoc), intent(out), optional :: meshloc
    type(ESMF_XGridSide_Flag), intent(out), optional :: xgridside
    integer, intent(out), optional :: gridindex
    integer, intent(out), optional :: gridToFieldMap(:)
    integer, intent(out), optional :: ungriddedLBound(:)
    integer, intent(out), optional :: ungriddedUBound(:)
    integer, intent(out), optional :: totalLWidth(:,:)
    integer, intent(out), optional :: totalUWidth(:,:)
    integer, intent(out), optional :: localDeCount
    integer, intent(out), optional :: ssiLocalDeCount
    integer, intent(out), optional :: localDeToDeMap(:)
    integer, intent(out), optional :: minIndex(:)
    integer, intent(out), optional :: maxIndex(:)
    integer, intent(out), optional :: elementCount(:)
    integer, intent(out), optional :: localMinIndex(:)
    integer, intent(out), optional :: localMaxIndex(:)
    integer, intent(out), optional :: localElementCount(:)
    logical, intent(out), optional :: isESMFAllocated
    character(len=*), intent(out), optional :: name
    type(ESMF_VM), intent(out), optional :: vm
    integer, intent(out), optional :: rc
!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersionExceptions{5.2.0r}
! \item\apiStatusModifiedSinceVersion{5.2.0r}
! \begin{description}
! \item[6.3.0r] Added argument {\tt vm} in order to offer information about the
! VM on which the Field was created.
! \item[8.1.0] Added argument {\tt minIndex}.
! The new argument allows the user to query the global lower bounds of the field data across all PETs.\newline
! Added argument {\tt maxIndex}.
! The new argument allows the user to query the global upper bounds of the field data across all PETs.\newline
! Added argument {\tt elementCount}.
! The new argument allows the user to query the global number of items of the field data across all PETs.\newline
! Added argument {\tt localMinIndex}.
! The new argument allows the user to query the PET local lower bounds globally indexed of the field data.\newline
! Added argument {\tt localMaxIndex}.
! The new argument allows the user to query the PET local upper bounds globally indexed of the field data.\newline
! Added argument {\tt localElementCount}.
! The new argument allows the user to query the PET local number of items of the field data.\newline
! Added argument {\tt ssiLocalDeCount} and {\tt localarrayList} to
! support DE sharing between PETs on the same single system
! image (SSI).\newline
! Added argument {\tt localDeToDeMap} to support DE handling from
! the Field level rather than require user to go to Array level.
! \item[8.5.0] Added argument {\tt isESMFAllocated} to support user query of
! whether ESMF or user is responsible for the data allocation
! referenced by this object.
! \item[8.6.0] Added argument {\tt geom} to support user query of Geom object.\newline
! Added argument {\tt geomDimCount} for easy access to dimension of
! underlying Geom object.\newline
! Added argument {\tt ungriddedDimCount} for easy access to the
! number of ungridded dimensions in the Field object.\newline
! Added argument {\tt replicatedDimCount} for easy access to the
! number of dimensions across which the Field object is replicated.
! \end{description}
! \end{itemize}
!
! !DESCRIPTION:
! Query an {\tt ESMF\_Field} object for various pieces of information.
! All arguments after the {\tt field} argument are optional. To select
! individual items use the {\tt named\_argument=value} syntax. For an example
! and associated documentation using this method see section
! \ref{sec:field:usage:field_get_default}.
!
! The arguments are:
! \begin{description}
! \item [field]
! {\tt ESMF\_Field} object to query.
! \item[{[arrayspec]}]
! {\tt ESMF\_ArraySpec} object containing the type/kind/rank information
! of the Field object.
! \item [{[status]}]
! \begin{sloppypar}
! The status of the Field. See section \ref{const:fieldstatus} for a
! complete list of values.
! \end{sloppypar}
! \item [{[geomtype]}]
! The type of geometry on which the Field is built. See
! section~\ref{const:geomtype} for the range of values.
! \item [{[geom]}]
! {\tt ESMF\_Geom}.
! \item [{[grid]}]
! {\tt ESMF\_Grid}.
! \item [{[mesh]}]
! \apiStatusCompatibleException
! {\tt ESMF\_Mesh}.
! \item [{[locstream]}]
! \apiStatusCompatibleException
! {\tt ESMF\_LocStream}.
! \item [{[xgrid]}]
! \apiStatusCompatibleException
! {\tt ESMF\_XGrid}.
! \item [{[array]}]
! {\tt ESMF\_Array}.
! \item[{[localarrayList]}]
! Upon return this holds a list of the associated {\tt ESMC\_LocalArray}
! objects. {\tt localarrayList} must be allocated to be of size
! {\tt localDeCount} or {\tt ssiLocalDeCount}.
! \item [{[typekind]}]
! TypeKind specifier for Field. See section \ref{const:typekind} for a
! complete list of values.
! \item [{[rank]}]
! Number of dimensions in the physical memory of the {\tt field} data. It is
! identical to dimCount when the corresponding grid is a non-arbitrary grid.
! It is less than dimCount when the grid is arbitrarily distributed.
! For an detailed discussion of this parameter, please see
! section \ref{sec:field:usage:createArbGrid} and
! section \ref{sec:field:usage:createArbGridRep}.
! \item [{[dimCount]}]
! Number of geometrical dimensions in {\tt field}.
! For an detailed discussion of this parameter, please see
! section \ref{sec:field:usage:createArbGrid} and
! section \ref{sec:field:usage:createArbGridRep}.
! \item [{[geomDimCount]}]
! Number of dimensions in the underlying Geom object.
! \item [{[ungriddedDimCount]}]
! Number of ungridded dimensions in the Field object.
! \item [{[replicatedDimCount]}]
! Number of replicated dimensions in the Field object.
! \item [{[staggerloc]}]
! Stagger location of data in grid cells. For valid
! predefined values and interpretation of results see
! section \ref{const:staggerloc}.
! \item [{[meshloc]}]
! \begin{sloppypar}
! \apiStatusCompatibleException
! The part of the mesh to build the Field on. Can be either
! {\tt ESMF\_MESHLOC\_NODE} or {\tt ESMF\_MESHLOC\_ELEMENT}. If not set,
! defaults to {\tt ESMF\_MESHLOC\_NODE}.
! \end{sloppypar}
! \item [{[xgridside]}]
! \begin{sloppypar}
! \apiStatusCompatibleException
! The side of the XGrid that the Field was created on. See section
! \ref{const:xgridside} for a complete list of values.
! \end{sloppypar}
! \item [{[gridIndex]}]
! \apiStatusCompatibleException
! If xgridside is {\tt ESMF\_XGRIDSIDE\_A} or {\tt ESMF\_XGRIDSIDE\_B}
! then this index tells which Grid/Mesh on that side the Field was
! created on.
! \item [{[gridToFieldMap]}]
! List with number of elements equal to the
! {\tt grid}'s dimCount. The list elements map each dimension
! of the {\tt grid} to a dimension in the {\tt field} by
! specifying the appropriate {\tt field} dimension index. The default is to
! map all of the {\tt grid}'s dimensions against the lowest dimensions of
! the {\tt field} in sequence, i.e. {\tt gridToFieldMap} = (/1,2,3,.../).
! The total ungridded dimensions in the {\tt field}
! are the total {\tt field} dimensions less
! the dimensions in
! the {\tt grid}. Ungridded dimensions must be in the same order they are
! stored in the {\t field}.
! \item [{[ungriddedLBound]}]
! Lower bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedLBound} argument must be
! equal or greater to {\tt ungriddedDimCount}, the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. Note that the ordering of
! the ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[ungriddedUBound]}]
! Upper bounds of the ungridded dimensions of the {\tt field}.
! The number of elements in the {\tt ungriddedUBound} argument must be
! equal or greater to {\tt ungriddedDimCount}, the number of ungridded
! dimensions in the {\tt field}. All ungridded dimensions of the
! {\tt field} are also undistributed. Note that the ordering of
! the ungridded dimensions is the same as their order in the {\tt field}.
! \item [{[totalLWidth]}]
! Lower bound of halo region. The size of the first dimension of this array is the number
! of gridded dimensions in the {\tt field}. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalLWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should be max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! The size of the 2nd dimension of this array is localDeCount.
! \item [{[totalUWidth]}]
! Upper bound of halo region. The size of the first dimension of this array is the number
! of gridded dimensions in the {\tt field}. However, ordering of the elements
! needs to be the same as they appear in the {\tt field}. Values default
! to 0. If values for totalUWidth are specified they must be reflected in
! the size of the {\tt field}. That is, for each gridded dimension the
! {\tt field} size should max( {\tt totalLWidth} + {\tt totalUWidth}
! + {\tt computationalCount}, {\tt exclusiveCount} ).
! The size of the 2nd dimension of this array is localDeCount.
! \item [{[localDeCount]}]
! Upon return this holds the number of PET-local DEs defined in
! the DELayout associated with the Field object.
! \item[{[ssiLocalDeCount]}]
! The number of DEs in the Field available to the local PET. This
! includes DEs that are local to other PETs on the same SSI, that are
! accessible via shared memory.
! \item[{[localDeToDeMap]}]
! Mapping between localDe indices and the (global) DEs associated with
! the local PET. The localDe index variables are discussed in sections
! \ref{DELayout_general_mapping} and \ref{Array_native_language_localde}.
! The provided actual argument must be of size {\tt localDeCount}, or
! {\tt ssiLocalDeCount}, and will be filled accordingly.
! \item[{[minIndex]}]
! Upon return this holds the global lower bounds of the field data across all PETs.
! This information will be identical across all PETs.
! {\tt minIndex} must be allocated to be of size equal to the field rank.
! \item[{[maxIndex]}]
! Upon return this holds the global upper bounds of the field data across all PETs.
! This information will be identical across all PETs.
! {\tt maxIndex} must be allocated to be of size equal to the field rank.
! \item[{[elementCount]}]
! Upon return this holds the global number of items of the field data across all PETs.
! This information will be identical across all PETs.
! {\tt elementCount} must be allocated to be of size equal to the field rank.
! \item[{[localMinIndex]}]
! Upon return this holds the PET local lower bounds globally indexed of the field data.
! {\tt localMinIndex} must be allocated to be of size equal to the field rank.
! \item[{[localMaxIndex]}]
! Upon return this holds the PET local upper bounds globally indexed of the field data.
! {\tt localMaxIndex} must be allocated to be of size equal to the field rank.
! \item[{[localElementCount]}]
! Upon return this holds the PET local number of items of the field data.
! {\tt localElementCount} must be allocated to be of size equal to the field rank.
! \item[{[isESMFAllocated]}]
! Set to {\tt .true.} for data allocations held by ESMF. Set to
! {\tt .false.} otherwise.
! \item[{[name]}]
! Name of the Field object.
! \item[{[vm]}]
! The VM on which the Field object was created.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    type(ESMF_FieldType), pointer :: ftype
    integer :: localrc, fieldrank
    type(ESMF_Status) :: fieldstatus
    type(ESMF_DistGrid) :: distgrid

    ! Initialize
    localrc = ESMF_RC_NOT_IMPL
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! check variables
    ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc)

    ! Get Field pointer
    ftype => field%ftypep

   call ESMF_BaseGetStatus(ftype%base, fieldstatus, rc=localrc)
   if (ESMF_LogFoundError(localrc, &
     ESMF_ERR_PASSTHRU, &
     ESMF_CONTEXT, rcToReturn=rc)) return

    if(present(arrayspec)) then
        call ESMF_ArrayGet(ftype%array, arrayspec=arrayspec, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if(present(status)) then
      status = ftype%status
    endif

    ! Get the geometry type
    if (present(geomtype)) then
        call ESMF_GeomGet(ftype%geom, geomtype=geomtype, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(geom)) then
      geom = ftype%geom
    endif

    if (present(grid)) then
        if (ftype%status .ne. ESMF_FIELDSTATUS_GRIDSET .and. &
            ftype%status .ne. ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No Grid or invalid Grid attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_GeomGet(ftype%geom, &
                  grid=grid, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(staggerloc)) then
        if (ftype%status .ne. ESMF_FIELDSTATUS_GRIDSET .and. &
            ftype%status .ne. ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No Grid or invalid Grid attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_GeomGet(ftype%geom, &
                  staggerloc=staggerloc, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(meshloc)) then
        if (ftype%status .ne. ESMF_FIELDSTATUS_GRIDSET .and. &
            ftype%status .ne. ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No Mesh or invalid Mesh attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_GeomGet(ftype%geom, &
                  meshloc=meshloc, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(mesh)) then
        if (ftype%status .ne. ESMF_FIELDSTATUS_GRIDSET .and. &
            ftype%status .ne. ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No Mesh or invalid Mesh attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_GeomGet(ftype%geom, &
                  mesh=mesh, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(locstream)) then
        if (ftype%status .ne. ESMF_FIELDSTATUS_GRIDSET .and. &
            ftype%status .ne. ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No LocStream or invalid LocStream attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_GeomGet(ftype%geom, &
                  locstream=locstream, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(xgrid)) then
        if (ftype%status .ne. ESMF_FIELDSTATUS_GRIDSET .and. &
            ftype%status .ne. ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No XGrid or invalid XGrid attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_GeomGet(ftype%geom, &
                  xgrid=xgrid, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(xgridSide)) then
        if (ftype%status .ne. ESMF_FIELDSTATUS_GRIDSET .and. &
            ftype%status .ne. ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No XGrid or invalid XGrid attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_GeomGet(ftype%geom, &
                  xgridside=xgridside, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(gridIndex)) then
        if (ftype%status .ne. ESMF_FIELDSTATUS_GRIDSET .and. &
            ftype%status .ne. ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No XGrid or invalid XGrid attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_GeomGet(ftype%geom, &
                  gridIndex=gridIndex, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(array)) then
        if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="No data attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        array = ftype%array
    endif

    if (present(gridToFieldMap)) then
        if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="Cannot get gridToFieldMap", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        gridToFieldMap = ftype%gridToFieldMap(1:size(gridToFieldMap))
    endif

    if (present(ungriddedLBound)) then
        if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="Cannot get ungriddedLBound", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        ungriddedLBound = ftype%ungriddedLBound(1:size(ungriddedLBound))
    endif

    if (present(ungriddedUBound)) then
        if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="Cannot get ungriddedUBound", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        ungriddedUBound = ftype%ungriddedUBound(1:size(ungriddedUBound))
    endif

    if (present(totalLWidth)) then
        !totalLWidth = ftype%totalLWidth(1:size(totalLWidth))
        call ESMF_ArrayGet(ftype%array, totalLWidth=totalLWidth, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(totalUWidth)) then
        !totalUWidth = ftype%totalUWidth(1:size(totalUWidth))
        call ESMF_ArrayGet(ftype%array, totalUWidth=totalUWidth, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(typekind)) then
        if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="Cannot return typekind because no data attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_ArrayGet(ftype%array, &
                           typekind=typekind, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(rank) &
      .or.present(ungriddedDimCount).or.present(replicatedDimCount)) then
      if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
        if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
          msg="Cannot access requested information, no data attached to Field", &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif
      call ESMF_ArrayGet(ftype%array, rank=rank, &
        undistDimCount=ungriddedDimCount, replicatedDimCount=replicatedDimCount, &
        rc=localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(dimCount)) then
        if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="Cannot return dimCount because no data attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        dimCount = ftype%dimCount
    endif

    if (present(geomDimCount)) then
      call ESMF_GeomGet(ftype%geom, dimCount=geomDimCount, rc=localrc)
      if (ESMF_LogFoundError(localrc, &
        ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if (present(localDeCount) &
      .or.present(ssiLocalDeCount) &
      .or.present(localDeToDeMap) &
      .or.present(localarrayList)) then
        if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
            if (ESMF_LogFoundError(ESMF_RC_OBJ_BAD, &
              msg="Cannot return information because no data attached to Field", &
              ESMF_CONTEXT, rcToReturn=rc)) return
        endif
        call ESMF_ArrayGet(ftype%array, localDeCount=localDeCount, &
          ssiLocalDeCount=ssiLocalDeCount, localDeToDeMap=localDeToDeMap, &
          localarrayList=localarrayList, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
    endif

    if(present(minIndex) .or. present(maxIndex) .or. present(elementCount) .or. &
         present(localMinIndex) .or. present(localMaxIndex) .or. present(localElementCount)) then

       call ESMF_ArrayGet(ftype%array, rank=fieldrank, &
            distgrid=distgrid, rc=localrc)
       if (localrc .ne. ESMF_SUCCESS) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
               msg="Cannot retrieve fieldrank from ftypep%array", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif

       if(present(minIndex) .or. present(maxIndex) .or. &
            present(elementCount)) then
          call getGlobalIndexInfo(ftype=ftype, distgrid=distgrid, fieldrank=fieldrank, &
               minIndex=minIndex, maxIndex=maxIndex, elementCount=elementCount, &
               rc=localrc)
          if (ESMF_LogFoundError(localrc, &
               ESMF_ERR_PASSTHRU, &
               ESMF_CONTEXT, rcToReturn=rc)) return
       endif

       if(present(localminIndex) .or. present(localmaxIndex) .or. &
            present(localelementCount) ) then
          call getLocalIndexInfo(ftype=ftype, distgrid=distgrid, fieldrank=fieldrank, &
               localMinIndex=localMinIndex, localMaxIndex=localMaxIndex, localElementCount=localElementCount, &
               rc=localrc)
          if (ESMF_LogFoundError(localrc, &
               ESMF_ERR_PASSTHRU, &
               ESMF_CONTEXT, rcToReturn=rc)) return
       endif

    endif ! present(minIndex or maxIndex or elementCount or localMinIndex or localMaxIndex or localElementCount)

    if (present(name)) then
      if (field%isNamedAlias) then
        name = trim(field%name)
      else
        call ESMF_GetName(ftype%base, name, localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
      endif
    endif

    if (present(vm)) then
        call ESMF_GetVM(ftype%base, vm, localrc)
        if (ESMF_LogFoundError(localrc, &
          ESMF_ERR_PASSTHRU, &
          ESMF_CONTEXT, rcToReturn=rc)) return
        ESMF_INIT_CHECK_DEEP(ESMF_VMGetInit, vm, rc)
    endif

    ! Obtain dealloc information
    if (present(isESMFAllocated)) then
      isESMFAllocated = .false.
      if (ftype%status /= ESMF_FIELDSTATUS_COMPLETE) then
        call ESMF_LogSetError(ESMF_RC_OBJ_BAD, &
          msg="No data attached to Field", &
          ESMF_CONTEXT, rcToReturn=rc)
        return
      endif
      call ESMF_ArrayGet(ftype%array, isESMFAllocated=isESMFAllocated, &
        rc=localrc)
      if (ESMF_LogFoundError(localrc, ESMF_ERR_PASSTHRU, &
        ESMF_CONTEXT, rcToReturn=rc)) return
    end if

    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldGetDefault
!------------------------------------------------------------------------------

#undef ESMF_METHOD
#define ESMF_METHOD "getGlobalIndexInfo"
! Helper method to determine various global index information for an ESMF_Field
  subroutine getGlobalIndexInfo(ftype, distgrid, fieldrank, &
       minIndex, maxIndex, elementCount, &
       rc)

    type(ESMF_FieldType), intent(in) :: ftype
    type(ESMF_DistGrid), intent(in) :: distgrid
    integer, intent(in) :: fieldrank
    integer, intent(out), optional :: minIndex(:)
    integer, intent(out), optional :: maxIndex(:)
    integer, intent(out), optional :: elementCount(:)
    integer, intent(out), optional :: rc

    integer :: localrc
    integer, dimension(:), allocatable :: l_min, l_max, l_elecount
    integer :: i, j, k
    integer :: dg_dimcount, tilecount
    integer, allocatable :: dg2fmap(:)
    logical, allocatable :: isDist(:)
    integer, allocatable :: minIndexPTile(:,:)
    integer, allocatable :: maxIndexPTile(:,:)

    ! initialize return code; assume routine not implemented
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    allocate(l_min(fieldrank), l_max(fieldrank), l_elecount(fieldrank))
    l_min = -1
    l_max = -1
    l_elecount = -1

    if(present(minIndex)) then
       if(size(minIndex) .ne. fieldrank) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="the size of minIndex must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif
    if(present(maxIndex)) then
       if(size(maxIndex) .ne. fieldrank) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="the size of maxIndex must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif
    if(present(elementCount)) then
       if(size(elementCount) .ne. fieldrank) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="the size of elementCount must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif

    call ESMF_DistGridGet(distgrid, &
         dimcount=dg_dimcount, tilecount=tilecount, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    allocate(dg2fmap(dg_dimcount))
    allocate(isDist(fieldrank))

    dg2fmap = -1
    isDist = .false.
    do i = 1, dg_dimcount
       dg2fmap(i) = ftype%gridToFieldMap(i)
       isDist(dg2fmap(i)) = .true.
    enddo
    allocate(minIndexPTile(dg_dimcount, tilecount), maxIndexPTile(dg_dimcount, tileCount))

    call ESMF_DistGridGet(distgrid, &
         minIndexPTile=minIndexPTile, maxIndexPTile = maxIndexPTile, &
         rc=localrc)
    if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    if(tilecount /= 1) then
       call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
            msg="Cannot retrieve global information for more than 1 tile yet.", &
            ESMF_CONTEXT, rcToReturn=rc)
       return
    endif

    ! Calculate global bounds
    j = 1
    k = 1
    do i = 1, fieldrank
       if(isDist(i)) then
          ! Global Global, this will be identical on all pets
          ! - note hard-coding of tile=1 (hence the assumption of tilecount == 1,
          ! enforced above)
          l_min(dg2fmap(j)) = minIndexPTile(j, 1)
          l_max(dg2fmap(j)) = maxIndexPTile(j, 1)
          l_elecount(dg2fmap(j)) = l_max(dg2fmap(j)) - l_min(dg2fmap(j)) + 1

          j = j + 1
       else
          ! Global Global, this will be identical on all pets
          l_min(i) = ftype%ungriddedLBound(k)
          l_max(i) = ftype%ungriddedUBound(k)
          l_elecount(i) = l_max(i) - l_min(i) + 1

          k = k + 1
       endif
    enddo
    deallocate(dg2fmap, minIndexPTile, maxIndexPTile)
    deallocate(isDist)

    if(present(minIndex)) minIndex(1:fieldrank) = l_min
    if(present(maxIndex)) maxIndex(1:fieldrank) = l_max
    if(present(elementCount)) elementCount(1:fieldrank) = l_max - l_min + 1
    deallocate(l_min, l_max, l_elecount)

    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine getGlobalIndexInfo

#undef ESMF_METHOD
#define ESMF_METHOD "getLocalIndexInfo"
! Helper method to determine various local index information for an ESMF_Field
  subroutine getLocalIndexInfo(ftype, distgrid, fieldrank, &
       localMinIndex, localMaxIndex, localElementCount, &
       rc)

    type(ESMF_FieldType), intent(in) :: ftype
    type(ESMF_DistGrid), intent(in) :: distgrid
    integer, intent(in) :: fieldrank
    integer, intent(out), optional :: localMinIndex(:)
    integer, intent(out), optional :: localMaxIndex(:)
    integer, intent(out), optional :: localElementCount(:)
    integer, intent(out), optional :: rc

    integer :: localrc
    integer, dimension(:), allocatable :: ll_min, ll_max, ll_elecount
    integer :: i, j, k
    integer :: decount, ldecount, dg_dimcount
    integer, allocatable :: dg2fmap(:)
    logical, allocatable :: isDist(:)
    integer, allocatable :: minIndexPDe(:,:)
    integer, allocatable :: maxIndexPDe(:,:)
    integer, allocatable :: ldeToDeMap(:)

    ! initialize return code; assume routine not implemented
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    allocate(ll_min(fieldrank), ll_max(fieldrank), ll_elecount(fieldrank))
    ll_min = -1
    ll_max = -1
    ll_elecount = -1

    if(present(localminIndex)) then
       if(size(localminIndex) .ne. fieldrank) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="the size of localMinIndex must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif
    if(present(localmaxIndex)) then
       if(size(localmaxIndex) .ne. fieldrank) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="the size of localMaxIndex must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif
    if(present(localElementCount)) then
       if(size(localElementCount) .ne. fieldrank) then
          call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
               msg="the size of localElementCount must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
          return
       endif
    endif

    call ESMF_DistGridGet(distgrid, localdecount=ldecount, &
         decount=decount, dimcount=dg_dimcount, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    allocate(dg2fmap(dg_dimcount))
    allocate(isDist(fieldrank))

    dg2fmap = -1
    isDist = .false.
    do i = 1, dg_dimcount
       dg2fmap(i) = ftype%gridToFieldMap(i)
       isDist(dg2fmap(i)) = .true.
    enddo
    allocate(ldeToDeMap(1))
    allocate(minIndexPDe(dg_dimcount, decount), maxIndexPDe(dg_dimcount, deCount))

    call ESMF_DistGridGet(distgrid, localdeToDeMap=lDeToDeMap, &
         minIndexPDe=minIndexPDe, maxIndexPDe=maxIndexPDe, rc=localrc)
    if (ESMF_LogFoundError(localrc, &
         ESMF_ERR_PASSTHRU, &
         ESMF_CONTEXT, rcToReturn=rc)) return

    if(ldecount /= 1) then
       call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
            msg="Only implemented for field with 1 localDe.", &
            ESMF_CONTEXT, rcToReturn=rc)
       return
    endif

    ! Calculate global bounds
    j = 1
    k = 1
    do i = 1, fieldrank
       if(isDist(i)) then

          ! Global bounds on local PET
          ! - note hard-coding of DE=1 (hence the assumption of ldecount == 1,
          ! enforced above)
          ll_min(dg2fmap(j)) = minIndexPDe(j, ldeToDeMap(1)+1)
          ll_max(dg2fmap(j)) = maxIndexPDe(j, ldeToDeMap(1)+1)
          ll_elecount(dg2fmap(j)) = ll_max(dg2fmap(j)) - ll_min(dg2fmap(j)) + 1

          j = j + 1
       else

          ! Global bounds on local PET
          ll_min(i) = ftype%ungriddedLBound(k)
          ll_max(i) = ftype%ungriddedUBound(k)
          ll_elecount(i) = ll_max(i) - ll_min(i) + 1

          k = k + 1
       endif
    enddo
    deallocate(dg2fmap, minIndexPDe, maxIndexPDe)
    deallocate(ldeToDeMap, isDist)

    if(present(localminIndex)) localminIndex(1:fieldrank) = ll_min
    if(present(localmaxIndex)) localmaxIndex(1:fieldrank) = ll_max
    if(present(localelementCount)) localelementCount(1:fieldrank) = ll_max - ll_min + 1
    deallocate(ll_min, ll_max, ll_elecount)

    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine getLocalIndexInfo
!------------------------------------------------------------------------------
!------------------------------------------------------------------------------ 
! <This section created by macro - do not edit directly> 
 
!! < start of macros which become actual subroutine bodies after expansion > 
 
!---------------------------------------------------------------------------- 
!BOP 
! !IROUTINE: ESMF_FieldGet - Get a DE-local Fortran array pointer from a Field 
! 
! !INTERFACE: 
! ! Private name; call using ESMF_FieldGet() 
! subroutine ESMF_FieldGetDataPtr<rank><type><kind>(field, keywordEnforcer, localDe, & 
! farrayPtr, exclusiveLBound, exclusiveUBound, exclusiveCount, & 
! computationalLBound, computationalUBound, computationalCount, & 
! totalLBound, totalUBound, totalCount, rc) 
! 
! !ARGUMENTS: 
! type(ESMF_Field), intent(in) :: field 
!type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
! integer, intent(in), optional :: localDe 
! <type> (ESMF_KIND_<kind>), pointer :: farrayPtr(<rank>) 
! integer, intent(out), optional :: exclusiveLBound(:) 
! integer, intent(out), optional :: exclusiveUBound(:) 
! integer, intent(out), optional :: exclusiveCount(:) 
! integer, intent(out), optional :: computationalLBound(:) 
! integer, intent(out), optional :: computationalUBound(:) 
! integer, intent(out), optional :: computationalCount(:) 
! integer, intent(out), optional :: totalLBound(:) 
! integer, intent(out), optional :: totalUBound(:) 
! integer, intent(out), optional :: totalCount(:) 
! integer, intent(out), optional :: rc 
! 
! !STATUS: 
! \begin{itemize} 
! \item\apiStatusCompatibleVersion{5.2.0r} 
! \end{itemize} 
! 
! !DESCRIPTION: 
! Get a Fortran pointer to DE-local memory allocation within {\tt field}. 
! For convenience DE-local bounds can be queried at the same time. 
! For an example and 
! associated documentation using this method see section 
! \ref{sec:field:usage:field_get_dataptr}. 
! 
! The arguments are: 
! \begin{description} 
! \item [field] 
! {\tt ESMF\_Field} object. 
! \item[{[localDe]}] 
! Local DE for which information is requested. {\tt [0,..,localDeCount-1]}. 
! For {\tt localDeCount==1} the {\tt localDe} argument may be omitted, 
! in which case it will default to {\tt localDe=0}. 
! \item [farrayPtr] 
! Fortran array pointer which will be pointed at DE-local memory allocation. 
! It depends on the specific entry point 
! of {\tt ESMF\_FieldCreate()} used during {\tt field} creation, which 
! Fortran operations are supported on the returned {\tt farrayPtr}. See 
! \ref{Field:rest} for more details. 
! \item[{[exclusiveLBound]}] 
! Upon return this holds the lower bounds of the exclusive region. 
! {\tt exclusiveLBound} must be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \item[{[exclusiveUBound]}] 
! Upon return this holds the upper bounds of the exclusive region. 
! {\tt exclusiveUBound} must be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \item[{[exclusiveCount]}] 
! Upon return this holds the number of items, {\tt exclusiveUBound-exclusiveLBound+1}, 
! in the exclusive region per dimension. 
! {\tt exclusiveCount} must 
! be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \item[{[computationalLBound]}] 
! \begin{sloppypar} 
! Upon return this holds the lower bounds of the computational region. 
! {\tt computationalLBound} must be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \end{sloppypar} 
! \item[{[computationalUBound]}] 
! \begin{sloppypar} 
! Upon return this holds the lower bounds of the computational region. 
! {\tt computationalUBound} must be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \end{sloppypar} 
! \item[{[computationalCount]}] 
! Upon return this holds the number of items in the computational region per dimension 
! (i.e. {\tt computationalUBound-computationalLBound+1}). {\tt computationalCount} must 
! be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \item[{[totalLBound]}] 
! Upon return this holds the lower bounds of the total region. 
! {\tt totalLBound} must be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \item[{[totalUBound]}] 
! Upon return this holds the lower bounds of the total region. 
! {\tt totalUBound} must be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \item[{[totalCount]}] 
! \begin{sloppypar} 
! Upon return this holds the number of items in the total region per dimension 
! (i.e. {\tt totalUBound-totalLBound+1}). {\tt computationalCount} must 
! be allocated to be of size equal to {\tt field}'s {\tt dimCount}. 
! See section \ref{Array_regions_and_default_bounds} for a description 
! of the regions and their associated bounds and counts. 
! \end{sloppypar} 
! \item [{[rc]}] 
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors. 
! \end{description} 
! 
!EOP 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_INTEGER_1_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr1Di1(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr1Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr2Di1(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr2Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr3Di1(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr3Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr4Di1(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr4Di1 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr5Di1(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr5Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr6Di1(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr6Di1 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr7Di1(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i1), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr7Di1 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
#ifndef ESMF_NO_INTEGER_2_BYTE 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr1Di2(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr1Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr2Di2(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr2Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr3Di2(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr3Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr4Di2(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr4Di2 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr5Di2(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr5Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr6Di2(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr6Di2 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr7Di2(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i2), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr7Di2 
!---------------------------------------------------------------------------- 
 
#endif 
#endif 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr1Di4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr1Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr1Di8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr1Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr1Dr4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr1Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr1Dr8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr1Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr2Di4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr2Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr2Di8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr2Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr2Dr4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr2Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr2Dr8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr2Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr3Di4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr3Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr3Di8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr3Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr3Dr4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr3Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr3Dr8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr3Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr4Di4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr4Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr4Di8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr4Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr4Dr4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr4Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr4Dr8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr4Dr8 
!---------------------------------------------------------------------------- 
 
#ifndef ESMF_NO_GREATER_THAN_4D 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr5Di4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr5Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr5Di8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr5Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr5Dr4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr5Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr5Dr8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr5Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr6Di4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr6Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr6Di8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr6Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr6Dr4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr6Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr6Dr8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr6Dr8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr7Di4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr7Di4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr7Di8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 integer (ESMF_KIND_i8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr7Di8 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr7Dr4(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r4), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr7Dr4 
!---------------------------------------------------------------------------- 
 
!---------------------------------------------------------------------------- 
#undef ESMF_METHOD 
#define ESMF_METHOD "ESMF_FieldGetDataPtr" 
 subroutine ESMF_FieldGetDataPtr7Dr8(field, & 
 localDe, farrayPtr, keywordEnforcer, & 
 exclusiveLBound, exclusiveUBound, exclusiveCount, & 
 computationalLBound, computationalUBound, computationalCount, & 
 totalLBound, totalUBound, totalCount, rc) 

 ! input arguments 
 type(ESMF_Field), intent(in) :: field 
 integer, intent(in), optional :: localDe 
 real (ESMF_KIND_r8), dimension(:,:,:,:,:,:,:), pointer :: farrayPtr 
 type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below 
 integer, intent(out), optional :: exclusiveLBound(:) 
 integer, intent(out), optional :: exclusiveUBound(:) 
 integer, intent(out), optional :: exclusiveCount(:) 
 integer, intent(out), optional :: computationalLBound(:) 
 integer, intent(out), optional :: computationalUBound(:) 
 integer, intent(out), optional :: computationalCount(:) 
 integer, intent(out), optional :: totalLBound(:) 
 integer, intent(out), optional :: totalUBound(:) 
 integer, intent(out), optional :: totalCount(:) 
 integer, intent(out), optional :: rc 

 ! local variables 
 integer :: localrc 

 if (present(rc)) rc = ESMF_RC_NOT_IMPL 
 localrc = ESMF_RC_NOT_IMPL 

 ! check variables 
 ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc) 

 call ESMF_ArrayGet(field%ftypep%array, localDe=localDe, & 
 farrayPtr=farrayPtr, rc=localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 if (present(exclusiveLBound) .or. & 
 present(exclusiveUBound) .or. & 
 present(exclusiveCount) .or. & 
 present(computationalLBound) .or. & 
 present(computationalUBound) .or. & 
 present(computationalCount) .or. & 
 present(totalLBound) .or. & 
 present(totalUBound) .or. & 
 present(totalCount)) then 

 call ESMF_FieldGetBounds(field, localDe=localDe, & 
 exclusiveLBound=exclusiveLBound, exclusiveUBound=exclusiveUBound, & 
 exclusiveCount=exclusiveCount, & 
 computationalLBound=computationalLBound, & 
 computationalUBound=computationalUBound, & 
 computationalCount=computationalCount, & 
 totalLBound=totalLBound, totalUBound=totalUBound, & 
 totalCount=totalCount, & 
 rc = localrc) 

 if (ESMF_LogFoundError(localrc, & 
 ESMF_ERR_PASSTHRU, & 
 ESMF_CONTEXT, rcToReturn=rc)) return 

 endif 

 if (present(rc)) rc = ESMF_SUCCESS 

 end subroutine ESMF_FieldGetDataPtr7Dr8 
!---------------------------------------------------------------------------- 
 
#endif 
 
! < end macro - do not edit directly > 
!------------------------------------------------------------------------------ 



!------------------------------------------------------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldGetBounds"
!BOP
! !IROUTINE: ESMF_FieldGetBounds - Get DE-local Field data bounds

! !INTERFACE:
  ! Private name; call using ESMF_FieldGetBounds()
  subroutine ESMF_FieldGetBounds(field, keywordEnforcer, localDe, &
    exclusiveLBound, exclusiveUBound, exclusiveCount, computationalLBound, &
    computationalUBound, computationalCount, totalLBound, &
    totalUBound, totalCount, rc)

! !ARGUMENTS:
    type(ESMF_Field), intent(in) :: field
type(ESMF_KeywordEnforcer), optional:: keywordEnforcer ! must use keywords below
    integer, intent(in), optional :: localDe
    integer, intent(out), optional :: exclusiveLBound(:)
    integer, intent(out), optional :: exclusiveUBound(:)
    integer, intent(out), optional :: exclusiveCount(:)
    integer, intent(out), optional :: computationalLBound(:)
    integer, intent(out), optional :: computationalUBound(:)
    integer, intent(out), optional :: computationalCount(:)
    integer, intent(out), optional :: totalLBound(:)
    integer, intent(out), optional :: totalUBound(:)
    integer, intent(out), optional :: totalCount(:)
    integer, intent(out), optional :: rc

!
! !STATUS:
! \begin{itemize}
! \item\apiStatusCompatibleVersion{5.2.0r}
! \end{itemize}
!
! !DESCRIPTION:
! This method returns the bounds information of a field that consists of a
! internal grid and a internal array. The exclusive and computational bounds
! are shared between the grid and the array but the total bounds are the array
! bounds plus the halo width. The count is the number of elements between each
! bound pair.
!
! The arguments are:
! \begin{description}
! \item[field]
! Field to get the information from.
! \item[{[localDe]}]
! Local DE for which information is requested. {\tt [0,..,localDeCount-1]}.
! For {\tt localDeCount==1} the {\tt localDe} argument may be omitted,
! in which case it will default to {\tt localDe=0}.
! \item[{[exclusiveLBound]}]
! Upon return this holds the lower bounds of the exclusive region.
! {\tt exclusiveLBound} must be allocated to be of size equal to the field rank.
! Please see section~\ref{sec:grid:usage:bounds} for a description
! of the regions and their associated bounds and counts.
! \item[{[exclusiveUBound]}]
! Upon return this holds the upper bounds of the exclusive region.
! {\tt exclusiveUBound} must be allocated to be of size equal to the field rank.
! Please see section~\ref{sec:grid:usage:bounds} for a description
! of the regions and their associated bounds and counts.
! \item[{[exclusiveCount]}]
! Upon return this holds the number of items, {\tt exclusiveUBound-exclusiveLBound+1},
! in the exclusive region per dimension.
! {\tt exclusiveCount} must
! be allocated to be of size equal to the field rank.
! Please see section~\ref{sec:grid:usage:bounds} for a description
! of the regions and their associated bounds and counts.
! \item[{[computationalLBound]}]
! Upon return this holds the lower bounds of the stagger region.
! {\tt computationalLBound} must be allocated to be of size equal to the field rank.
! Please see section~\ref{sec:grid:usage:bounds} for a description
! of the regions and their associated bounds and counts.
! \item[{[computationalUBound]}]
! Upon return this holds the upper bounds of the stagger region.
! {\tt computationalUBound} must be allocated to be of size equal to the field rank.
! Please see section~\ref{sec:grid:usage:bounds} for a description
! of the regions and their associated bounds and counts.
! \item[{[computationalCount]}]
! Upon return this holds the number of items in the computational region per dimension
! (i.e. {\tt computationalUBound-computationalLBound+1}). {\tt computationalCount}
! must be allocated to be of size equal to the field rank.
! Please see section~\ref{sec:grid:usage:bounds} for a description
! of the regions and their associated bounds and counts.
! \item[{[totalLBound]}]
! Upon return this holds the lower bounds of the total region.
! {\tt totalLBound} must be allocated to be of size equal to the field rank.
! \item[{[totalUBound]}]
! Upon return this holds the upper bounds of the total region.
! {\tt totalUBound} must be allocated to be of size equal to the field rank.
! \item[{[totalCount]}]
! \begin{sloppypar}
! Upon return this holds the number of items in the total region per dimension
! (i.e. {\tt totalUBound-totalLBound+1}). {\tt totalCount} must
! be allocated to be of size equal to the field rank.
! \end{sloppypar}
! \item[{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOP
!------------------------------------------------------------------------------
    integer :: gridrank, fieldrank, localrc
    integer :: i, j
    logical :: filled(ESMF_MAXDIM)

    integer :: localDeCount, dimCount, l_localDe, dimCount_repdim
    integer, allocatable :: arrayBnd(:, :), arrayCompBnd(:,:), arrayExclBnd(:,:)
    integer, allocatable :: undistBnd(:)
    integer, allocatable :: distgridToArrayMap(:)
    integer, allocatable :: distgridToPackedArrayMap(:)
    integer, dimension(:), allocatable :: l_elb, l_eub, l_clb, l_cub
    integer, dimension(:), allocatable :: l_tlb, l_tub
    integer, dimension(:), allocatable :: l_ec, l_cc, l_tc


    if (present(rc)) rc = ESMF_RC_NOT_IMPL
    localrc = ESMF_RC_NOT_IMPL

    ! check variables
    ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc)

    call ESMF_GeomGet(field%ftypep%geom, dimCount=gridrank, rc=localrc)
    if (localrc .ne. ESMF_SUCCESS) then
       call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg="Cannot retrieve gridrank from ftypep%grid", &
           ESMF_CONTEXT, rcToReturn=rc)
       return
    endif
    call ESMF_ArrayGet(field%ftypep%array, rank=fieldrank, &
        localDeCount=localDeCount, &
        rc=localrc)
    if (localrc .ne. ESMF_SUCCESS) then
       call ESMF_LogSetError(rcToCheck=ESMF_RC_OBJ_BAD, &
          msg="Cannot retrieve fieldrank from ftypep%array", &
           ESMF_CONTEXT, rcToReturn=rc)
       return
    endif

    if(localDeCount .gt. 1 .and. (.not. present(localDe))) then
       call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
          msg="localDe must be present when localDeCount is greater than 1", &
           ESMF_CONTEXT, rcToReturn=rc)
       return
    endif
    if(present(localDe)) then
        l_localDe = localDe
    else
        l_localDe = 0
    endif

    if(present(exclusiveLBound)) then
        if(size(exclusiveLBound) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of exclusiveLBound must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif
    if(present(exclusiveUBound)) then
        if(size(exclusiveUBound) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of exclusiveUBound must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif
    if(present(exclusiveCount)) then
        if(size(exclusiveCount) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of exclusiveCount must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif
    if(present(computationalLBound)) then
        if(size(computationalLBound) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of computationalLBound must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif
    if(present(computationalUBound)) then
        if(size(computationalUBound) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of computationalUBound must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif
    if(present(computationalCount)) then
        if(size(computationalCount) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of computationalCount must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif
    if(present(totalLBound)) then
        if(size(totalLBound) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of totalLBound must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif
    if(present(totalUBound)) then
        if(size(totalUBound) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of totalUBound must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif
    if(present(totalCount)) then
        if(size(totalCount) .ne. fieldrank) then
           call ESMF_LogSetError(rcToCheck=ESMF_RC_ARG_VALUE, &
              msg="the size of totalCount must equal to the field rank", &
               ESMF_CONTEXT, rcToReturn=rc)
           return
        endif
    endif

    ! local temporaries to store return values
    allocate(l_elb(fieldrank), l_eub(fieldrank), l_clb(fieldrank), l_cub(fieldrank))
    allocate(l_tlb(fieldrank), l_tub(fieldrank))
    allocate(l_ec(fieldrank), l_cc(fieldrank), l_tc(fieldrank))

    ! count the number of ungridded dimension in Grid to Field mapping
    dimCount_repdim = 0
    do i = 1, gridrank
        if(field%ftypep%gridToFieldMap(i) == 0) dimCount_repdim = dimCount_repdim + 1
    enddo

    if (present(totalLBound) .or. present(totalUBound) .or. &
        present(computationalLBound) .or. present(computationalUBound) .or. &
        present(exclusiveLBound) .or. present(exclusiveUBound)) then

        ! Get Size of Array info
        call ESMF_ArrayGet(field%ftypep%array, dimCount=dimCount, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! Allocate storage for Array info
        ! size of undistBnd is total number of undistributed dimensions
        ! including replicated dimensions
        allocate(arrayBnd(dimCount, 0:localDeCount-1))
        allocate(arrayCompBnd(dimCount, 0:localDeCount-1))
        allocate(arrayExclBnd(dimCount, 0:localDeCount-1))
        allocate(undistBnd(fieldRank-dimCount+dimCount_repdim))
        allocate(distgridToPackedArrayMap(dimCount))
        allocate(distgridToArrayMap(dimCount))

        ! get distributed dimension map
        call ESMF_ArrayGet(field%ftypep%array, &
            distgridToPackedArrayMap=distgridToPackedArrayMap, &
            distgridToArrayMap=distgridToArrayMap, rc=localrc)
        if (ESMF_LogFoundError(localrc, &
            ESMF_ERR_PASSTHRU, &
            ESMF_CONTEXT, rcToReturn=rc)) return

        ! fill lower Bounds
        if (present(totalLBound) .or. present(computationalLBound) &
            .or. present(exclusiveLBound)) then
            call ESMF_ArrayGet(field%ftypep%array, totalLBound=arrayBnd, &
                computationalLBound=arrayCompBnd, exclusiveLBound=arrayExclBnd, &
                undistLBound=undistBnd, rc=localrc)
            if (ESMF_LogFoundError(localrc, &
                ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return

            ! set distributed dimensions
            filled=.false.
            do i=1, dimCount
                if(distgridToPackedArrayMap(i) /= 0) then
                   l_tlb(distgridToArrayMap(i))=&
                    arrayBnd(distgridToPackedArrayMap(i),l_localDe)
                   l_clb(distgridToArrayMap(i))=&
                    arrayCompBnd(distgridToPackedArrayMap(i),l_localDe)
                   l_elb(distgridToArrayMap(i))=&
                    arrayExclBnd(distgridToPackedArrayMap(i),l_localDe)
                   filled(distgridToArrayMap(i))=.true.
                endif
            enddo

            ! set undistributed dimensions
            j=1
            do i=1, fieldRank
               if (.not. filled(i)) then
                   l_tlb(i)=undistBnd(j)
                   l_clb(i)=undistBnd(j)
                   l_elb(i)=undistBnd(j)
                   j=j+1;
               endif
            enddo
        endif

        ! fill upper bounds
        if (present(totalUBound) .or. present(computationalUBound) &
            .or. present(exclusiveUBound)) then
            call ESMF_ArrayGet(field%ftypep%array, totalUBound=arrayBnd, &
                computationalUBound=arrayCompBnd, exclusiveUBound=arrayExclBnd, &
                undistUBound=undistBnd, rc=localrc)
            if (ESMF_LogFoundError(localrc, &
                ESMF_ERR_PASSTHRU, &
                ESMF_CONTEXT, rcToReturn=rc)) return

            ! set distributed dimensions
            filled=.false.
            do i=1, dimCount
                if(distgridToPackedArrayMap(i) /= 0) then
                    l_tub(distgridToArrayMap(i))=&
                        arrayBnd(distgridToPackedArrayMap(i),l_localDe)
                    l_cub(distgridToArrayMap(i))=&
                        arrayCompBnd(distgridToPackedArrayMap(i),l_localDe)
                    l_eub(distgridToArrayMap(i))=&
                        arrayExclBnd(distgridToPackedArrayMap(i),l_localDe)
                    filled(distgridToArrayMap(i))=.true.
                endif
            enddo

            ! set undistributed dimensions
            j=1
            do i=1, fieldRank
               if (.not. filled(i)) then
                   l_tub(i)=undistBnd(j)
                   l_cub(i)=undistBnd(j)
                   l_eub(i)=undistBnd(j)
                   j=j+1;
               endif
           enddo
        endif

        ! deallocate data
        deallocate(arrayBnd)
        deallocate(arrayCompBnd)
        deallocate(arrayExclBnd)
        deallocate(undistBnd)
        deallocate(distgridToPackedArrayMap)
        deallocate(distgridToArrayMap)
    endif

    do i = 1, fieldrank
        l_cc(i) = l_cub(i) - l_clb(i) + 1
        l_ec(i) = l_eub(i) - l_elb(i) + 1
        l_tc(i) = l_tub(i) - l_tlb(i) + 1
    end do

    ! prepare return values
    ! computational region
    if(present(computationalLBound)) &
        computationalLBound(1:fieldrank) = l_clb
    if(present(computationalUBound)) &
        computationalUBound(1:fieldrank) = l_cub
    if(present(computationalCount)) &
        computationalCount(1:fieldrank) = l_cc

    ! exclusive region
    if(present(exclusiveLBound)) &
        exclusiveLBound(1:fieldrank) = l_elb
    if(present(exclusiveUBound)) &
        exclusiveUBound(1:fieldrank) = l_eub
    if(present(exclusiveCount)) &
        exclusiveCount(1:fieldrank) = l_ec

    ! total region
    if(present(totalLBound)) &
        totalLBound(1:fieldrank) = l_tlb
    if(present(totalUBound)) &
        totalUBound(1:fieldrank) = l_tub
    if(present(totalCount)) &
        totalCount(1:fieldrank) = l_tc

    ! Deallocate temporary arrays
    deallocate(l_elb, l_eub, l_clb, l_cub)
    deallocate(l_tlb, l_tub)
    deallocate(l_ec, l_cc, l_tc)

    if (present(rc)) rc = ESMF_SUCCESS
  end subroutine ESMF_FieldGetBounds
!------------------------------------------------------------------------------

! -------------------------- ESMF-private method -------------------------------
#undef ESMF_METHOD
#define ESMF_METHOD "ESMF_FieldGetTimestamp"
!BOPI
! !IROUTINE: ESMF_FieldGetTimestamp - Get timestamp on Field

! !INTERFACE:
  subroutine ESMF_FieldGetTimestamp(field, timestamp, rc)
!
! !ARGUMENTS:
    type(ESMF_Field), intent(in) :: field
    integer, intent(out), optional :: timestamp(10)
    integer, intent(out), optional :: rc
!
! !DESCRIPTION:
! Query an {\tt ESMF\_Field} for its timestamp.
!
! The arguments are:
! \begin{description}
! \item [field]
! {\tt ESMF\_Field} object to query.
! \item [timestamp]
! Timestamp, an array of 10 integer values.
! \item [{[rc]}]
! Return code; equals {\tt ESMF\_SUCCESS} if there are no errors.
! \end{description}
!EOPI
!------------------------------------------------------------------------------
    ! initialize return code; assume routine not implemented
    if (present(rc)) rc = ESMF_RC_NOT_IMPL

    ! check variables
    ESMF_INIT_CHECK_DEEP(ESMF_FieldGetInit,field,rc)

    ! Get timestamp
    timestamp(:) = field%ftypep%timestamp(:)

    if (present(rc)) rc = ESMF_SUCCESS

  end subroutine ESMF_FieldGetTimestamp
!------------------------------------------------------------------------------

end module ESMF_FieldGetMod
