#! /bin/bash

## "@(#) $Id: fposs.fixINSfilename 211657 2017-11-16 23:16:17Z jpritcha $"

################################################################################
## usage definition
usage() {
  printf "\
Usage: ${execName} [-h|Dvnq] <INS-file>

  -h|--help: help
  -D|--debug: debug
  -n|--dryRun: dry-run
  -q|--quiet: quiet
     -q Only Warnings and Errors are written out (to stderr)
     -q -q increases quietness, only Errors are written out (to stderr)
     -q -q -q increases quietness, NO output to stdout or stderr
  -v|--verbose: verbose

  <INS-file>: the <INS-file> will be renamed to it's default standard
     filename
"
  exit ${exstat:-0}
}

################################################################################
## Definition of variable who need default values
execName="`basename ${0}`"
quietLevel=0

################################################################################
## Generic utility functions

################################################################################
## Useful environment variables
if [ -z "${USD_FUNCTIONS_SED}" ]; then
  USD_FUNCTIONS_SED="sed"
  which ssed > /dev/null 2>&1 && USD_FUNCTIONS_SED="ssed"
  which gsed > /dev/null 2>&1 && USD_FUNCTIONS_SED="gsed"
  export USD_FUNCTIONS_SED
fi
if [ -z "${USD_FUNCTIONS_DATE}" ]; then
  USD_FUNCTIONS_DATE="date"
  which gdate > /dev/null 2>&1 && USD_FUNCTIONS_DATE="gdate"
  export USD_FUNCTIONS_DATE
fi

################################################################################
verboseLog() {
  if [ ! -z "${vbose}" ]
  then
    echo "${execName}:: $1" >> ${2:-/dev/stdout}
  fi
}

################################################################################
debugLog() {
  if [ ! -z "${debug}" ]
  then
    echo "${execName}::debug:: $1" >> ${2:-/dev/stdout}
  fi
}

################################################################################
notQuietLog() {
  if [ -z "${quiet}" ]
  then
    echo "${execName}:: $1" >> ${2:-/dev/stdout}
  fi
}

################################################################################
errorLog() {
  if [ ${quietLevel:-0} -lt 3 ]
  then
    echo "${execName}::Error:: $1" >> ${2:-/dev/stderr}
  fi
}

################################################################################
warningLog() {
  if [ ${quietLevel:-0} -lt 2 ]
  then
    echo "${execName}::Warning:: $1" >> ${2:-/dev/stderr}
  fi
}

################################################################################
execCmd() {
  if [ ! -z "${debug}" ] || [ ! -z "${dryRun}" ] || [ ! -z "${ECdebug}" ] || [ ! -z "${ECdryRun}" ]
  then
    echo "${execName}::execCmd:: $1"
  fi
  unset execCmd_PIPESTATUS
  if [ -z "${dryRun}"  ] && [ -z "${ECdryRun}" ] || ${ignoreDryRun:-false}
  then
    unsetIgnoreDryRun="${ignoreDryRun:-true}"
    unset ignoreDryRun
    local i=0 S
    if echo "${1}" | grep '&[[:space:]]*$' >/dev/null 2>&1 ; then
      eval "${1}"
      return 0
    else
      eval "${1} ; for S in \${PIPESTATUS[@]} ; do execCmd_PIPESTATUS[\$i]=\$S ; (( i++ )) ; done"
    fi
    ${unsetIgnoreDryRun:-false} && unset ignoreDryRun
    return ${execCmd_PIPESTATUS[${#execCmd_PIPESTATUS[@]}]}
  fi
  execCmd_PIPESTATUS=0
  return 0
}

################################################################################
execCmdDebug() {
  ECdebug="--debug"
  execCmd "${1}"
  retval=$?
  unset ECdebug
  return $retval
}

################################################################################
execCmdDryRun() {
  ECdryRun="--dryRun"
  execCmd "${1}"
  retval=$?
  unset ECdryRun
  return $retval
}

################################################################################
execCmdIgnoreDryRun() {
  unsetIgnoreDryRun="${ignoreDryRun:-true}"
  ignoreDryRun="true"
  execCmd "${1}"
  retval=$?
  ${unsetIgnoreDryRun:-false} && unset ignoreDryRun
  return $retval
}

################################################################################
execCmdExitOnError() {
  execCmd "${1}"
  local i=1 j S estat=0
  for S in ${execCmd_PIPESTATUS[@]} ; do
    if [ $S -ne 0 ]; then
      if [ ${#execCmd_PIPESTATUS[@]} -gt 1 ]; then
        unset preDel postDel 
        (( j=i+1 ))
        (( k=i-1 ))
        [ $i -gt 1 ] && preDel=" -e 1,${k}d"
        [ $i -lt ${#execCmd_PIPESTATUS[@]} ] && postDel=" -e ${j},${#execCmd_PIPESTATUS[@]}d"
        errorLog "'$(echo ${1} | tr '|' '\n' | sed${preDel}${postDel} -e 's/^ *//' -e 's/ *$//')', exited with status $S"
      else
        errorLog "'${1}' failed, exited with status $S"
      fi
      estat=$S
      (( i++ ))
    fi
  done
  [ $estat -ne 0 ] && exit $estat
}

################################################################################
## Usage:
## checkVarIsSet <VAR-NAME> <level> [error-message] [exit-status]
##      <VAR-NAME>: name of variable to check for, no '$'
##         <level>: fatal|fatalNoExit|warn
## [error-message]: Optional Error/Warning message to print
##   [exit-status]: Optional exit status value
##
checkVarIsSet(){
  v1="$`echo $1`"
  v2="`eval echo $v1`"
  debugLog "Variable $1 is set to $v2"
  if [ -z "${v2}" ]
  then
    debugLog "Variable $1 is not set"
    case $2 in
      fatal)       errorLog   "${3:-Variable $1 not set}"; exstat=${4:-1} ; usage ;;
      fatalNoExit) errorLog   "${3:-Variable $1 not set}"; (( exstat=${exstat-0}+${4:-1} ));;
      warn)        warningLog "${3:-Variable $1 not set}";;
      *)           errorLog   "*** Coding error:: error-level not recognised"; exit 1
    esac
  else
    debugLog "Variable $1 is set to $v2"
  fi
}
################################################################################

################################################################################
## Local of functions definitions

################################################################################
## Command line options
while [ ! -z "${1}" ]; do

  case $1 in
    -h|--help)    usage ; shift;;
    -D|--debug)   debug="-D"; quiet=""; vbose="-v";  quietLevel=0 ; shift;;
    -n|--dryRun)  dryRun="-n"; shift;;
    -q|--quiet)   quiet="${quiet} -q"; vbose=""; debug=""; (( quietLevel++ )) ; shift;;
    -v|--verbose) vbose="-v"; quiet="";  quietLevel=0 ; shift;;
    *)            INSs="${INSs:+${INSs} }${1}"; shift;;
  esac
done

################################################################################
## Main loop starts...

exstat=0

# Call other template.bash based scripts with command line options:
##  ${quiet:+ -q}${dryRun:+ -n}${debug:+ -D}${vbose:+ -v}
# or
##  ${quiet:+ -q}${dryRun:+ -n}${debug:+ -d}${vbose:+ -v}
# for older scripts with "-d" for debug rather than -D

checkVarIsSet INSs fatal

for INS in ${INSs} ; do
  PAFID=$(grep ^PAF.ID "${INS}" | sed -e 's/^.*[[:space:]]"//' -e 's/".*$//')
  execCmd "mv ${vbose:+-v} ""${INS}"" ""${PAFID}.ins"""
done

exit ${exstat:-0}
