/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { useState, useEffect, useCallback } from 'react';
import { AuthType, loadApiKey, debugLogger, } from '@google/gemini-cli-core';
import { getErrorMessage } from '@google/gemini-cli-core';
import { AuthState } from '../types.js';
import { validateAuthMethod } from '../../config/auth.js';
export function validateAuthMethodWithSettings(authType, settings) {
    const enforcedType = settings.merged.security.auth.enforcedType;
    if (enforcedType && enforcedType !== authType) {
        return `Authentication is enforced to be ${enforcedType}, but you are currently using ${authType}.`;
    }
    if (settings.merged.security.auth.useExternal) {
        return null;
    }
    // If using Gemini API key, we don't validate it here as we might need to prompt for it.
    if (authType === AuthType.USE_GEMINI) {
        return null;
    }
    return validateAuthMethod(authType);
}
export const useAuthCommand = (settings, config, initialAuthError = null) => {
    const [authState, setAuthState] = useState(initialAuthError ? AuthState.Updating : AuthState.Unauthenticated);
    const [authError, setAuthError] = useState(initialAuthError);
    const [apiKeyDefaultValue, setApiKeyDefaultValue] = useState(undefined);
    const onAuthError = useCallback((error) => {
        setAuthError(error);
        if (error) {
            setAuthState(AuthState.Updating);
        }
    }, [setAuthError, setAuthState]);
    const reloadApiKey = useCallback(async () => {
        const envKey = process.env['GEMINI_API_KEY'];
        if (envKey !== undefined) {
            setApiKeyDefaultValue(envKey);
            return envKey;
        }
        const storedKey = (await loadApiKey()) ?? '';
        setApiKeyDefaultValue(storedKey);
        return storedKey;
    }, []);
    useEffect(() => {
        if (authState === AuthState.AwaitingApiKeyInput) {
            // eslint-disable-next-line @typescript-eslint/no-floating-promises
            reloadApiKey();
        }
    }, [authState, reloadApiKey]);
    useEffect(() => {
        // eslint-disable-next-line @typescript-eslint/no-floating-promises
        (async () => {
            if (authState !== AuthState.Unauthenticated) {
                return;
            }
            const authType = settings.merged.security.auth.selectedType;
            if (!authType) {
                if (process.env['GEMINI_API_KEY']) {
                    onAuthError('Existing API key detected (GEMINI_API_KEY). Select "Gemini API Key" option to use it.');
                }
                else {
                    onAuthError('No authentication method selected.');
                }
                return;
            }
            if (authType === AuthType.USE_GEMINI) {
                const key = await reloadApiKey(); // Use the unified function
                if (!key) {
                    setAuthState(AuthState.AwaitingApiKeyInput);
                    return;
                }
            }
            const error = validateAuthMethodWithSettings(authType, settings);
            if (error) {
                onAuthError(error);
                return;
            }
            const defaultAuthType = process.env['GEMINI_DEFAULT_AUTH_TYPE'];
            if (defaultAuthType &&
                !Object.values(AuthType).includes(defaultAuthType)) {
                onAuthError(`Invalid value for GEMINI_DEFAULT_AUTH_TYPE: "${defaultAuthType}". ` +
                    `Valid values are: ${Object.values(AuthType).join(', ')}.`);
                return;
            }
            try {
                await config.refreshAuth(authType);
                debugLogger.log(`Authenticated via "${authType}".`);
                setAuthError(null);
                setAuthState(AuthState.Authenticated);
            }
            catch (e) {
                onAuthError(`Failed to login. Message: ${getErrorMessage(e)}`);
            }
        })();
    }, [
        settings,
        config,
        authState,
        setAuthState,
        setAuthError,
        onAuthError,
        reloadApiKey,
    ]);
    return {
        authState,
        setAuthState,
        authError,
        onAuthError,
        apiKeyDefaultValue,
        reloadApiKey,
    };
};
//# sourceMappingURL=useAuth.js.map