/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
export function isNodeError(error) {
    return error instanceof Error && 'code' in error;
}
export function getErrorMessage(error) {
    if (error instanceof Error) {
        return error.message;
    }
    try {
        return String(error);
    }
    catch {
        return 'Failed to get error details';
    }
}
export class FatalError extends Error {
    exitCode;
    constructor(message, exitCode) {
        super(message);
        this.exitCode = exitCode;
    }
}
export class FatalAuthenticationError extends FatalError {
    constructor(message) {
        super(message, 41);
    }
}
export class FatalInputError extends FatalError {
    constructor(message) {
        super(message, 42);
    }
}
export class FatalSandboxError extends FatalError {
    constructor(message) {
        super(message, 44);
    }
}
export class FatalConfigError extends FatalError {
    constructor(message) {
        super(message, 52);
    }
}
export class FatalTurnLimitedError extends FatalError {
    constructor(message) {
        super(message, 53);
    }
}
export class FatalToolExecutionError extends FatalError {
    constructor(message) {
        super(message, 54);
    }
}
export class FatalCancellationError extends FatalError {
    constructor(message) {
        super(message, 130); // Standard exit code for SIGINT
    }
}
export class CanceledError extends Error {
    constructor(message = 'The operation was canceled.') {
        super(message);
        this.name = 'CanceledError';
    }
}
export class ForbiddenError extends Error {
}
export class UnauthorizedError extends Error {
}
export class BadRequestError extends Error {
}
export class ChangeAuthRequestedError extends Error {
    constructor() {
        super('User requested to change authentication method');
        this.name = 'ChangeAuthRequestedError';
    }
}
export function toFriendlyError(error) {
    if (error && typeof error === 'object' && 'response' in error) {
        const gaxiosError = error;
        const data = parseResponseData(gaxiosError);
        if (data && data.error && data.error.message && data.error.code) {
            switch (data.error.code) {
                case 400:
                    return new BadRequestError(data.error.message);
                case 401:
                    return new UnauthorizedError(data.error.message);
                case 403:
                    // It's import to pass the message here since it might
                    // explain the cause like "the cloud project you're
                    // using doesn't have code assist enabled".
                    return new ForbiddenError(data.error.message);
                default:
            }
        }
    }
    return error;
}
function parseResponseData(error) {
    // Inexplicably, Gaxios sometimes doesn't JSONify the response data.
    if (typeof error.response?.data === 'string') {
        try {
            return JSON.parse(error.response?.data);
        }
        catch {
            return undefined;
        }
    }
    return error.response?.data;
}
/**
 * Checks if an error is a 401 authentication error.
 * Uses structured error properties from MCP SDK errors.
 *
 * @param error The error to check
 * @returns true if this is a 401/authentication error
 */
export function isAuthenticationError(error) {
    // Check for MCP SDK errors with code property
    // (SseError and StreamableHTTPError both have numeric 'code' property)
    if (error && typeof error === 'object' && 'code' in error) {
        const errorCode = error.code;
        if (errorCode === 401) {
            return true;
        }
    }
    // Check for UnauthorizedError class (from MCP SDK or our own)
    if (error instanceof Error &&
        error.constructor.name === 'UnauthorizedError') {
        return true;
    }
    if (error instanceof UnauthorizedError) {
        return true;
    }
    // Fallback: Check for MCP SDK's plain Error messages with HTTP 401
    // The SDK sometimes throws: new Error(`Error POSTing to endpoint (HTTP 401): ...`)
    const message = getErrorMessage(error);
    if (message.includes('401')) {
        return true;
    }
    return false;
}
//# sourceMappingURL=errors.js.map