import { Thread } from '../Thread.mjs';

/**
 * Creates a thread from a `MessagePort` instance in the browser.
 *
 * @see https://developer.mozilla.org/en-US/docs/Web/API/MessagePort
 *
 * @example
 * import {ThreadMessagePort} from '@quilted/threads';
 *
 * const channel = new MessageChannel();
 *
 * const threadOne = new ThreadMessagePort(channel.port1);
 * const threadTwo = new ThreadMessagePort(channel.port2, {
 *   exports: {
 *     async sendMessage(message) {
 *       console.log(message);
 *     },
 *   },
 * });
 *
 * channel.port1.start();
 * channel.port2.start();
 *
 * await threadOne.imports.sendMessage('Hello world!');
 */
class ThreadMessagePort extends Thread {
  /**
   * Starts a thread wrapped around a `MessagePort` object, and returns the imports
   * of the thread.
   *
   * @example
   * ```ts
   * import {ThreadMessagePort} from '@quilted/threads';
   *
   * const channel = new MessageChannel();
   *
   * ThreadMessagePort.export(channel.port1, {
   *   async getMessage() {
   *     return 'Hello, world!';
   *   },
   * });
   *
   * const {getMessage} = ThreadMessagePort.import(channel.port2);
   *
   * channel.port1.start();
   * channel.port2.start();
   *
   * const message = await getMessage(); // 'Hello, world!'
   * ```
   */
  static import(port, options) {
    return new ThreadMessagePort(port, options).imports;
  }

  /**
   * Starts a thread wrapped around a `MessagePort` object, providing the second
   * argument as the exports of the thread.
   *
   * @example
   * ```ts
   * import {ThreadMessagePort} from '@quilted/threads';
   *
   * const channel = new MessageChannel();
   *
   * ThreadMessagePort.export(channel.port1, {
   *   async getMessage() {
   *     return 'Hello, world!';
   *   },
   * });
   *
   * const {getMessage} = ThreadMessagePort.import(channel.port2);
   *
   * channel.port1.start();
   * channel.port2.start();
   *
   * const message = await getMessage(); // 'Hello, world!'
   * ```
   */
  static export(port, exports, options) {
    new ThreadMessagePort(port, {
      ...options,
      exports
    });
  }
  constructor(port, options) {
    super(portToMessageTarget(port), options);
    this.port = port;
  }
}
function portToMessageTarget(port) {
  return {
    send(...args) {
      port.postMessage(...args);
    },
    listen(listener, {
      signal
    }) {
      port.addEventListener('message', event => {
        listener(event.data);
      }, {
        signal
      });
    }
  };
}

export { ThreadMessagePort, portToMessageTarget };
