///////////////////////////////////////////////////////////////////////////////
//
// File: CommDataType.h
//
// For more information, please see: http://www.nektar.info
//
// The MIT License
//
// Copyright (c) 2006 Division of Applied Mathematics, Brown University (USA),
// Department of Aeronautics, Imperial College London (UK), and Scientific
// Computing and Imaging Institute, University of Utah (USA).
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//
// Description: Describes data types (using MPI_Datatype if available)
//
///////////////////////////////////////////////////////////////////////////////

#ifndef NEKTAR_LIB_UTILITIES_COMMDATATYPE_H
#define NEKTAR_LIB_UTILITIES_COMMDATATYPE_H

#include <LibUtilities/BasicConst/NektarUnivTypeDefs.hpp>
#include <vector>

#ifdef NEKTAR_USE_MPI
#include <mpi.h>

namespace Nektar::LibUtilities
{
typedef MPI_Datatype CommDataType;
} // namespace Nektar::LibUtilities

#elif NEKTAR_USING_PETSC

namespace Nektar
{
namespace LibUtilities
{
typedef unsigned int CommDataType;
}
} // namespace Nektar

#else

namespace Nektar
{
namespace LibUtilities
{
typedef unsigned int CommDataType;

#ifndef MPI_CHAR
#define MPI_CHAR ((CommDataType)0x4c000101)
#endif

#ifndef MPI_INT
#define MPI_INT ((CommDataType)0x4c000405)
#endif

#ifndef MPI_UNSIGNED
#define MPI_UNSIGNED ((CommDataType)0x4c000406)
#endif

#ifndef MPI_LONG
#define MPI_LONG ((CommDataType)0x4c000807)
#endif

#ifndef MPI_UNSIGNED_LONG
#define MPI_UNSIGNED_LONG ((CommDataType)0x4c000808)
#endif

#ifndef MPI_LONG_LONG
#define MPI_LONG_LONG ((CommDataType)0x4c000809)
#endif

#ifndef MPI_UNSIGNED_LONG_LONG
#define MPI_UNSIGNED_LONG_LONG ((CommDataType)0x4c000819)
#endif

#ifndef MPI_FLOAT
#define MPI_FLOAT ((CommDataType)0x4c00040a)
#endif

#ifndef MPI_DOUBLE
#define MPI_DOUBLE ((CommDataType)0x4c00080b)
#endif

#ifndef MPI_LONG_DOUBLE
#define MPI_LONG_DOUBLE ((CommDataType)0x4c00100c)
#endif
} // namespace LibUtilities
} // namespace Nektar
#endif

namespace Nektar
{
template <typename Dim, typename DataType> class Array;

namespace LibUtilities
{
int CommDataTypeGetSize(CommDataType);

template <class T> class CommDataTypeTraits
{
public:
    LIB_UTILITIES_EXPORT static CommDataType &GetDataType();

    static void *GetPointer(T &val)
    {
        return &val;
    }
    static const void *GetPointer(const T &val)
    {
        return &val;
    }
    static int GetCount([[maybe_unused]] const T &val)
    {
        return 1;
    }

    const static bool IsVector = false;
};

/**
 * Partial specialisation for vectors
 */
template <class elemT> class CommDataTypeTraits<std::vector<elemT>>
{
public:
    static CommDataType &GetDataType()
    {
        return CommDataTypeTraits<elemT>::GetDataType();
    }
    static void *GetPointer(std::vector<elemT> &val)
    {
        ASSERTL1(!val.empty(),
                 "Vector cannot be empty when trying to use GetPointer to "
                 "access a pointer to the first element.");
        return &val[0];
    }
    static const void *GetPointer(const std::vector<elemT> &val)
    {
        ASSERTL1(!val.empty(),
                 "Vector cannot be empty when trying to use GetPointer to "
                 "access a pointer to the first element.");
        return &val[0];
    }
    static size_t GetCount(const std::vector<elemT> &val)
    {
        return val.size();
    }
    const static bool IsVector = true;
};

/**
 * Partial specialisation for vectors
 */
template <class elemT> class CommDataTypeTraits<Array<OneD, elemT>>
{
public:
    static CommDataType &GetDataType()
    {
        return CommDataTypeTraits<elemT>::GetDataType();
    }
    static void *GetPointer(Array<OneD, elemT> &val)
    {
        return val.data();
    }
    static const void *GetPointer(const Array<OneD, elemT> &val)
    {
        return val.data();
    }
    static size_t GetCount(const Array<OneD, elemT> &val)
    {
        return val.size();
    }
    const static bool IsVector = true;
};
} // namespace LibUtilities
} // namespace Nektar

#endif
